<?php

namespace WP_Defender\Model\Notification;

use WP_Defender\Model\Scan as Model_Scan;

class Malware_Report extends \WP_Defender\Model\Notification {
	protected $table = 'wd_malware_scanning_report';

	public function before_load() {
		$default = array(
			'slug'                 => 'malware-report',
			'title'                => __( 'Malware Scanning - Reporting', 'wpdef' ),
			'status'               => self::STATUS_DISABLED,
			'description'          => __( 'Automatically run regular scans of your website and email you reports.', 'wpdef' ),
			// @since 3.0.0 Fix 'Guest'-line.
			'in_house_recipients'  => is_user_logged_in() ? array( $this->get_default_user() ) : array(),
			'out_house_recipients' => array(),
			'type'                 => 'report',
			// @since 2.7.0 Add Scheduled Scanning to Malware settings.
			// Hide Schedule and 'dry_run' fields on UI but no delete for backward compatibility with Scan settings, the config structure and the Hub.
			'frequency'            => 'weekly',
			'day'                  => 'sunday',
			'day_n'                => '1',
			'time'                 => '4:00',
			'dry_run'              => false,
			'configs'              => array(
				'always_send' => false,
				'error_send'  => false,
				'template'    => array(
					'found'     => array(
						'subject' => __( 'Scan of {SITE_URL} complete. {ISSUES_COUNT} issues found.', 'wpdef' ),
						'body'    => __( 'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I\'ve finished scanning {SITE_URL} for vulnerabilities and I found {ISSUES_COUNT} issues that you should take a closer look at!
{ISSUES_LIST}

Stay Safe,
WP Defender
WPMU DEV Superhero', 'wpdef' ),
					),
					'not_found' => array(
						'subject' => __( 'Scan of {SITE_URL} complete. {ISSUES_COUNT} issues found.', 'wpdef' ),
						'body'    => __( 'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I\'ve finished scanning {SITE_URL} for vulnerabilities and I found nothing. Well done for running such a tight ship!

Keep up the good work! With regular security scans and a well-hardened installation you\'ll be just fine.

Stay safe,
WP Defender
WPMU DEV Superhero', 'wpdef' ),
					),
					'error'     => array(
						'subject' => __( 'Couldn’t scan {SITE_URL} for vulnerabilities. ', 'wpdef' ),
						'body'    => __(
							'Hi {USER_NAME},

WP Defender here, reporting back from the front.

I couldn’t scan {SITE_URL} for vulnerabilities. Please visit your site and run a manual scan.

Stay Safe,
WP Defender
WPMU DEV Superhero ', 'wpdef' ),
					),
				),
			),
		);
		$this->import( $default );
	}

	/**
	 * @return bool
	 * @throws \Exception
	 */
	public function maybe_send() {
		// @since 2.7.0 First, check "Scheduled Scan" is checked or not.
		if ( ! ( new \WP_Defender\Model\Setting\Scan() )->scheduled_scanning ) {
			return false;
		}

		if ( empty( $this->last_sent ) || empty( $this->est_timestamp ) ) {
			$this->last_sent     = $this->est_timestamp;
			$this->est_timestamp = $this->get_next_run()->getTimestamp();
			$this->save();
		}
		$time = apply_filters( 'defender_current_time_for_report', new \DateTime( 'now', wp_timezone() ) );
		// Testing.
		if ( defined( 'WP_DEFENDER_TESTING' ) && true === constant( 'WP_DEFENDER_TESTING' ) ) {
			return true;
		}
		// Check by 'always_send' will be after scanning is complete.
		return $time->getTimestamp() >= $this->est_timestamp;
	}

	/**
	 * Steps:
	 * 1) check the current active scan,
	 * 2) if there isn't active scan then check the latest scan,
	 * 3) save the report settings,
	 * 4) run new scan.
	*/
	public function send() {
		// The current status.
		$active = Model_Scan::get_active();
		if ( is_object( $active ) ) {
			// Scan is running.
			return;
		}
		// Check the latest scan.
		$model = Model_Scan::get_last();
		if ( ! is_object( $model ) ) {
			return;
		}
		// Save the report settings.
		$this->last_sent     = $this->est_timestamp;
		$this->est_timestamp = $this->get_next_run()->getTimestamp();
		$this->save();
		// Run scan.
		$scan = Model_Scan::create( true );
		if ( is_object( $scan ) && ! is_wp_error( $scan ) ) {
			wd_di()->get( \WP_Defender\Controller\Scan::class )->do_async_scan( 'report' );
		}
	}

	/**
	 * Define labels for settings key.
	 *
	 * @param  string|null $key
	 *
	 * @return string|array|null
	 */
	public function labels( $key = null ) {
		$labels = array(
			'report'             => __( 'Malware Scanning - Reporting', 'wpdef' ),
			'always_send'        => __( 'Send notifications when no issues are detected', 'wpdef' ),
			'report_subscribers' => __( 'Recipients', 'wpdef' ),
			'day'                => __( 'Day of', 'wpdef' ),
			'day_n'              => __( 'Day of', 'wpdef' ),
			'time'               => __( 'Time of day', 'wpdef' ),
			'frequency'          => __( 'Frequency', 'wpdef' ),
		);

		if ( ! is_null( $key ) ) {
			return $labels[ $key ] ?? null;
		}

		return $labels;
	}

	/**
	 * @return string
	 */
	public function get_next_run_for_hub() {
		// Todo: check it because Scheduled Scanning moves to Malware settings on v2.7.0.
		return self::STATUS_ACTIVE === $this->status
			? $this->persistent_hub_datetime_format( $this->est_timestamp )
			: __( 'Never', 'wpdef' );
	}

	/**
	 * Additional converting rules.
	 *
	 * @param  array $configs
	 *
	 * @return array
	 * @since 3.1.0
	 */
	public function type_casting( $configs ) {
		$configs['always_send'] = (bool) $configs['always_send'];
		$configs['error_send']  = (bool) $configs['error_send'];

		return $configs;
	}
}